
/* 
      Arduino Distortion Unit
      MIDI Program
      Uses MIDIUSB.h library to run MIDI commands over the USB cable
      Works with all Arduinos that have the ATmega32U4 processor.  
      These have HID capabilities.
      
      VoiceA is used to modulate the other three voices.  It uses the tone( ) function
      
      Voices B, C, and D are controlled from a USB MIDI keyboard.

      Switch 1 (lower) can be turned on to record up to 50 note events and times.
      Switch 2 (upper) can be turned on to play back the recorded notes.
      
*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      CONSTANTS and Variables
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
// For use on an Arduino Pro Micro.  

#include "MIDIUSB.h"

// 
// ANALOG INPUTS
//
const int POT1 = A0;  //Potentiometer pin numbers
const int POT2 = A1;
const int POT3 = A2;
const int POT4 = A3;
const int POT5 = A7;

int pot1 = 0;
int pot2 = 0;
int pot3 = 0;
int pot4 = 0;
int pot5 = 0;

//
//DIGIITAL SWITCHES
//  
const int S1 = 7;  //Switch pin numbers
const int S2 = 8;
const int S3 = 16;
const int S4 = 14;

boolean switch1 = 0;
boolean switch2 = 0;
boolean pushbutton1 = 0;
boolean pushbutton2 = 0;

//
// SYTHESIZER CONSTANTS
//

const int VOICEA = 5;  //Voice pin numbers
const int VOICEB = 4;
const int VOICEC = 2;
const int VOICED = 3;


const int LED1 = 9;  //Led pin numbers
const int LED2 = 10;

//tone() frequency values for equal temperament A440 MIDI NoteON commands.  Used for VoiceA

// Note the limitations of tone() which at 16mhz specifies a minimum frequency of 31hz - in other words, notes below
// B0 will play at the wrong frequency since the timer can't run that slowly!

unsigned long toneFreq[128] = {8, 9, 9, 10, 10, 11, 12, 12, 13, 14, 15, 15, 16, 17, 18, 19, 21, 
    22, 23, 24, 26, 28, 29, 31, 33, 35, 37, 39, 41, 44, 46, 49, 52, 55, 58, 62, 65, 69, 73, 78, 
    82, 87, 92, 98, 104, 110, 117, 123, 131, 139, 147, 156, 165, 175, 185, 196, 208, 220, 233, 
    247, 262, 277, 294, 311, 330, 349, 370, 392, 415, 440, 466, 494, 523, 554, 587, 622, 659, 
    698, 740, 784, 831, 880, 932, 988, 1047, 1109, 1175, 1245, 1319, 1397, 1480, 1568, 1661, 
    1760, 1865, 1976, 2093, 2217, 2349, 2489, 2637, 2794, 2960, 3136, 3322, 3520, 3729, 3951, 
    4186, 4435, 4699, 4978, 5274, 5588, 5920, 5920, 6645, 7040, 7459, 7902, 8372, 8870, 9397, 
    9956, 10548, 11175, 11840, 12544};

// Table for the countdown values used to set equal temperament frequecies for Voices B and C
// Values calculated in a spreadsheet:  (loop-Frequency)/(2 * MIDI-Note-Frequency)   

unsigned long freqTable[128] = {1389, 1312, 1238, 1169, 1103, 1042, 983, 928, 875, 826, 780, 
    736, 695, 656, 619, 584, 552, 521, 492, 464, 438, 413, 390, 368, 348, 328, 310, 292, 276, 
    260, 246, 232, 219, 207, 195, 184, 174, 164, 155, 146, 138, 130, 123, 116, 109, 103, 98, 
    92, 87, 82, 77, 73, 69, 65, 61, 58, 55, 52, 49, 46, 43, 41, 39, 37, 34, 33, 31, 29, 27, 
    26, 24, 23, 22, 20, 19, 18, 17, 16, 15, 14, 14, 13, 12, 12, 11, 10, 10, 9, 9, 8, 8, 7, 7, 
    6, 6, 6, 5, 5, 5, 5, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1};   

byte eventNotes[50] = {0};           //holds recorded Midi Input NoteOn
unsigned long eventTimes[50] = {0};  //holds recorded times of Notes
byte eventVelocity[50] = {0};        // holds recorded note velocities
byte eventVoice[50] = {0};           // holds recorded voice assignment of notes

unsigned long eventTime = 0;
unsigned long startTime = 0; // created by timestamp()
int currentTime = 0;

int  eventIndex = 0;  // holds position within the 4 record arrays
bool record = 0;
bool playback = 0;

//current note on's for Arduino pin voices A, B, and C.  Use note value 0 for "not on". 

byte CurrentNoteOn[3] = {0, 0, 0}; 

int freqA = 100;  //variables to set voice frequencies
int freqB = 100;
int freqC = 100;
int freqD = 100;

unsigned long deltamicro = 0;
unsigned long lastmicro = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      SETUP()
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


void setup() {

delay(1000);
tone(VOICEA, 64, 200);
delay(300);
tone(VOICEA, 164, 200);
  
pinMode(LED1, OUTPUT);    //turn on LED1 as power indicator
digitalWrite(LED1, HIGH);

pinMode(LED2, OUTPUT);
digitalWrite(LED2, LOW);


pinMode(VOICEA, OUTPUT);    //Lows enable distortion gates
digitalWrite(VOICEA, LOW);
pinMode(VOICEB, OUTPUT);
digitalWrite(VOICEB, LOW);
pinMode(VOICEC, OUTPUT);  
digitalWrite(VOICEC, LOW);
pinMode(VOICED, OUTPUT);  
digitalWrite(VOICED, LOW);                               

pinMode(S1, INPUT);  // Set up switch inputs with pullup resistors
digitalWrite(S1, HIGH);
pinMode(S2, INPUT);
digitalWrite(S2, HIGH);
pinMode(S3, INPUT);
digitalWrite(S3, HIGH);
pinMode(S4, INPUT);
digitalWrite(S4, HIGH);
                     
// Serial.begin(115200);   //Used to read loop time    

} //End of Setup

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      MAIN LOOP
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loop() {

//  deltamicro = micros() - lastmicro;   //Used to read loop time  (44usec with an arduino Mega board ??)
//  lastmicro = micros();                // Loop Frequency = 1/deltamicro  (deltamicro is in microseconds)

// Arduino's tone() function can only be used to set up a squarewave on one output - VoiceA.
// Two more voices are created from a fast loop time decrementing two "freq" values and toggling voice outputs
// when they reach zero.  The "freq" values determine the frequency of the voices.

//VoiceA ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                                                   
    
       tone(VOICEA, (50 + analogRead(POT5)));  // set frequency of VoiceA using tone, used as modulator
      

  //VoiceB ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      if (CurrentNoteOn[0] != 0) {
        --freqB;                                         // decrement Voice B freqB counter
        if (freqB <= 0){
            digitalWrite(VOICEB, !digitalRead(VOICEB));  //toggle Voice B pin when counter reaches zero
            freqB = freqTable[CurrentNoteOn[0]];         //reload counter from table
            
      }
        }
      else { digitalWrite(VOICEB, LOW); }  
            
  //VoiceC ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      if (CurrentNoteOn[1] != 0) {                       // same as above for  Voice C
        --freqC;
        if (freqC <= 0){
            digitalWrite(VOICEC, !digitalRead(VOICEC));
           freqC = freqTable[CurrentNoteOn[1]];
          
      }
        }   
      else { digitalWrite(VOICEC, LOW); }  
 //VoiceD ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

      if (CurrentNoteOn[2] != 0) {                       // same as above for  Voice D
        --freqD;
        if (freqD <= 0){
            digitalWrite(VOICED, !digitalRead(VOICED));
           freqD = freqTable[CurrentNoteOn[2]];
          
      }
        }          
      else { digitalWrite(VOICED, LOW); }  

//--------------------------------------------------------------------------------
//
//---------------Playback of Notes[] at Times []-----------------------------
//

      while (playback) {
       
           currentTime = dur();                    
           while (eventTimes[eventIndex] <= currentTime){ 
                       
           playEvent(eventNotes[eventIndex], eventVelocity[eventIndex], eventVoice[eventIndex]);
           
                        eventIndex = eventIndex + 1;
                        if (eventNotes[eventIndex] == 255){   // no more recorded notes                             
                              eventIndex = 0;              // rewind to beginning
                              timestamp();                              
                              break;
                        }  //  end of rewind 
           } // end of while              
                                break;                     
      } //end of while playback



//--------------------------------------------------------------------------------

//--------------switches used to start and stop record or playback -------------

//--------------------------------------------------------------------------------



                  if (digitalRead(S1) && !record) {  //start record                
                      playback = 0;
                      record = 1;
                      digitalWrite(LED2, HIGH);
                      eventIndex = 0;
                                for (int i=0; i < 50; i++){
                                    eventNotes[i] = 255;
                                    eventTimes[i] = 0;
                                }
                       timestamp();
                       
                  }

                   else if (!digitalRead(S1) && record){  //stop record
                      playback = 0;
                      record = 0;
                      digitalWrite(LED2, LOW);  
                  }                  
                  
                  else if (digitalRead(S2) && !playback){  //start playback
                      record = 0;
                      digitalWrite(LED2, HIGH);
                      playback = 1;
                      eventIndex = 0;
                      timestamp();                                                                 
                  }
                                  
                  else if (!digitalRead(S2) && playback){  //stop playback
                      record = 0;
                      digitalWrite(LED2, LOW);
                      playback = 0;
                      CurrentNoteOn[0] = 0;
                      CurrentNoteOn[1] = 0;
                      CurrentNoteOn[2] = 0;
                  }      










midiEventPacket_t rx = MidiUSB.read();

  switch (rx.header) {
    
    case 0:
                break; //No pending events
      
    case 0x9:  //NoteOn

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
   //   Serial.println(deltamicro);  //used to find the loop time in microseconds

   if (rx.byte3 == 0){    //  Note OFF, note velocity of zero.
    
              for (int x = 0; x < 3; x ++){    // search through the 3 Arduino pin voices
                
                    if (CurrentNoteOn[x] == rx.byte2){ // voice playing to turn off?

                                   switch (x) {           
                                   case 0:                   
                                     CurrentNoteOn[x] = 0;
                                     digitalWrite(VOICEB, LOW);
                                     if (record){ loadEventArrays(rx.byte2, rx.byte3, x); }                                                                               
                                     break;                                 
                                  case 1:                    
                                     CurrentNoteOn[x] = 0; 
                                     digitalWrite(VOICEC, LOW);
                                     if (record){ loadEventArrays(rx.byte2, rx.byte3, x); }                                                                                
                                     break;                                 
                                  case 2:                   
                                     CurrentNoteOn[x] = 0; 
                                     digitalWrite(VOICED, LOW);
                                     if (record){ loadEventArrays(rx.byte2, rx.byte3, x); }                                                                                      
                                     break;       
                                   }  // end of switch 
                                                                           
                     }  // end of Note check                                                                                                                     
                     }  // end of for loop                   
   }
   else{    //Note ON,  note velocity not zero
              for (int x = 0; x < 3; x ++){    // search through the 3 Arduino pin voices
                
                    if (CurrentNoteOn[x] == 0){      // this voice is available                                                                          
                            CurrentNoteOn[x] = rx.byte2;  //pitch 
                            if (record){ loadEventArrays(rx.byte2, rx.byte3, x); }                                                                                                                                                              
                            break;  //  break out of voice search loop   
                                                              
                     }  // end of Note On check                                                                                                                                        
                     }  // end of for loop
   }
            break;
  
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

       
    case 0x8: // Note Off

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    for (int x = 0; x < 3; x ++){     // search through the 3 AY voices

    if (CurrentNoteOn[x] == rx.byte2){    // which voice is playing the note to turn off?

                 switch (x) {           
                 case 0:                   
                   CurrentNoteOn[x] = 0;
                    digitalWrite(VOICEB, LOW);  
                   if (record){ loadEventArrays(rx.byte2, 0, 0); }      
                   break;                                 
                case 1:                    
                   CurrentNoteOn[x] = 0; 
                    digitalWrite(VOICEC, LOW);  
                   if (record){ loadEventArrays(rx.byte2, 0, 1); }          
                   break;                                 
                case 2:                   
                   CurrentNoteOn[x] = 0;
                    digitalWrite(VOICED, LOW);  
                   if (record){ loadEventArrays(rx.byte2, 0, 2); }            
                   break;       
                 }  // end of switch 
                                                                           
           }  // end of Note check

     
           }  // end of for loop
                break;
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~            
                
    case 0xB:  // control Change     
                break;
      
    default:
                 break;
      
  }
  
 
} 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      END OF MAIN LOOP
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      Event Functions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loadEventArrays(int note, int velocity, int voice){
  eventTimes[eventIndex] = dur();
  eventNotes[eventIndex] = note;
  eventVelocity[eventIndex] = velocity;
  eventVoice[eventIndex] = voice;
  eventIndex = eventIndex + 1;
}

void playEvent(int n, int v, int voice){
        switch (voice) {           
                 case 0:                   
                   if (v == 0) {CurrentNoteOn[0] = 0;}
                   else {CurrentNoteOn[0] = n;}             
                   break;                                 
                case 1:
                   if (v == 0) {CurrentNoteOn[1] = 0;}
                   else {CurrentNoteOn[1] = n;}                         
                   break;                                 
                case 2:
                  if (v == 0) {CurrentNoteOn[2] = 0;}
                   else {CurrentNoteOn[2] = n;}                            
                   break;       
                 }  // end of switch 
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//              TIMER  Loop Functions
//  ______________________________________________________________________________

void timestamp() {startTime = millis(); };
// store current time from the running clock millis()

unsigned long dur(){
return (millis() - startTime);
}
// returns the current time minus the last store timestamp

void waitTill(unsigned long msec) {
  while (dur() < msec) {};
}
//wait till the time duration from timestamp equals the given time in msec

//  ______________________________________________________________________________



 
