
/* 
                       Full functioning SID Synth Chip with MIDI IN

      ** MIDI Input Note ON and Note OFF handled by Callback function
         2 voice polyphony with Voices 1 and 2.
      ** Slider1 - frequency offset for Voices 1 
      ** Slider2 - frequency offset for Voices 2 
      ** Slider3 - frequency offset for Modulating Voice 3 (course tune) 
      ** Slider4 - Pulse width Voice 1, 
      ** Slider5 - Pulse width Voice 2, Voice3 frequency Fine
      ** Slider6 - not used
     
      ** Tap bottom of SoftPot - Voice 1 gets waveform from switches and sliders 5 & 6
                                 ADSR envelope from Sliders 1,2,3,4
         Tap Middle of SoftPot - Voice 2 gets waveform from switches and sliders 5 & 6
                                 ADSR envelope from Sliders 1,2,3,4
         Tap Top of SoftPot - Get from switches - Voice1 Sync, Ring Mod and Voice 3 Sync, Ring Mod
                              Get from Sliders 1,2,3 - Filter type HP, BP, LP  (on or off)
                              Get from Sliders 4,5,6 - Filter on Voice 1,2,3, External (on or off)
      
      ** Switches 1-4 set the waveform type (noise, pulse, saw, triangle) 
         RINGMOD on slider5, SYNC on slider6, for voices 1 and 2.
         Values entered with softpot.
         
      ** PotX sets filter cutoff frequency, PotY sets filter resonance 
      ** 3OFF = 1, no modulation voice 3 in output
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  
*/
//____________________________________________________
//                      CONSTANTS and Variables
//____________________________________________________

#include <MIDI.h>  // version 4.x.x
MIDI_CREATE_INSTANCE(HardwareSerial, Serial1, MIDI); 

// User created MIDI Callback functions here

 const int  MIDI_TX = 18;  //on SERIAL1
 const int MIDI_RX = 19;  //on SERIAL1
// 
// ANALOG INPUTS
//
const int Slider1 = 0;
const int Slider2 = 1;
const int Slider3 = 2;
const int Slider4 = 3;
const int Slider5 = 4;
const int Slider6 = 5;
const int SoftPot = 6;
const int FrontPot = 7;
const int LightSensor1 = 8;
const int LightSensor2 = 9;
const int LightSensor3 = 10;
const int LightSensor4 = 11;
const int LightSensor5 = 12;
const int LightSensor6 = 13;

int slider1 = 0;
int slider2 = 0;
int slider3 = 0;
int slider4 = 0;
int slider5 = 0;
int slider6 = 0;
int softPot = 0;
int frontPot = 0;
int lightSensor1 = 0;
int lightSensor2 = 0;
int lightSensor3 = 0;
int lightSensor4 = 0;
int lightSensor5 = 0;
int lightSensor6 = 0;

//
//DIGIITAL SWITCHES
//
const int Switch1 = 14;
const int Switch2 = 15;
const int Switch3 = 16;
const int Switch4 = 17;

boolean switch1 = 0;
boolean switch2 = 0;
boolean switch3 = 0;
boolean switch4 = 0;



const int VoiceD1 = 7;
const int VoiceD2 = 8;
const int VoiceD3 = 9;

//
//SID SYNTHESIZER CONSTANTS
//
const int SIDCS = 5;
const int SIDRW = 6;

//SID read register values

int potX = 0;
int potY = 0;
int osc3_rand = 0;
int env3 = 0;

int addr[4] = {0, 0, 7, 14};  //voice register address offsets 1, 2, 3
int v = 0;

// voice register values (Use voice = 1, 2, or 3.  Don't use zero)

int Attack[4] = {0, 0, 0, 0};      // (0 to 15)
int Decay[4] = {0, 0, 0, 0};       // (0 to 15)
int Sustainx[4] = {0, 15, 15, 15};  // (0 to 15)
int Release[4] = {0, 0, 0, 0};     // (0 to 15)
int FreqLo[4] = {0, 0, 0, 0};      // (0 to 255)
int FreqHi[4] = {0, 16, 16, 16};   // (0 to 255)
int PulseWLo[4] = {0, 0, 0, 0};    // (0 to 255)
int PulseWHi[4] = {0, 8, 8, 8};    // (0 to 15)
int Waveshape[4] = {0, 0, 0, 0};   // Load with the bit values below

// bit values for the voice Waveshape (Control) register

const int NOISE = 128;
const int PULSE = 64;
const int SAWTOOTH = 32;
const int TRIANGLE = 16;
const int TEST = 8;
const int RINGMOD = 4;
const int SYNC = 2;

// bit values for filt of ldResFilt, add the ones you want, zero for none

const int FILTEX = 8;  // send external signal through the Filter
const int FILT3 = 4;   // send Voice 3 through the Filter
const int FILT2 = 2;   // send Voice 2 through the Filter
const int FILT1 = 1;   // send Voice 3 through the Filter

int filt = 0;

// bit values for mode of ldModeVol, add the ones you want, zero for none
  
const int OFF3 = 128;  // no Voice3 in the output (when used in ring modulation)
const int HP = 64;     // set Filter tos High Pass
const int BP = 32;     // set Filter to BandPass
const int LP = 16;     // set Filter to LowPass
bool hp = 0;
bool bp = 0;
bool lp = 0;
int res = 0;

const int LED = 20;

//SID frequency values for equal temperment A440 MIDI NoteON commands

byte SIDMidiNoteHi[128] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 
  2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 4, 
  4, 4, 4, 5, 5, 5, 6, 6, 6, 7, 7, 8, 
  8, 9, 9, 10, 10, 11, 12, 12, 13, 14, 15, 16, 
  17, 18, 19, 20, 21, 22, 24, 25, 27, 28, 30, 32, 
  34, 36, 38, 40, 43, 45, 48, 51, 54, 57, 61, 64, 
  68, 72, 76, 81, 86, 91, 96, 102, 108, 115, 122, 129, 
  137, 145, 153, 163, 172, 183, 193, 205, 217, 230, 244, 244, 
  244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 
  244, 244, 244, 244, 244, 244, 244, 244
};

byte SIDMidiNoteLow[128] = {
  137, 145, 154, 163, 173, 183, 194, 206, 218, 231, 244, 3, 
  18, 35, 52, 70, 90, 110, 132, 155, 180, 205, 233, 6, 
  37, 69, 104, 141, 179, 220, 8, 54, 103, 155, 210, 12, 
  73, 139, 208, 25, 103, 185, 16, 108, 206, 54, 163, 24, 
  147, 21, 160, 50, 205, 114, 32, 217, 156, 107, 71, 47, 
  38, 43, 63, 100, 155, 228, 64, 177, 56, 215, 141, 95, 
  75, 86, 127, 200, 53, 199, 128, 98, 112, 173, 27, 189, 
  151, 171, 253, 145, 107, 142, 255, 196, 225, 90, 54, 122, 
  45, 86, 250, 34, 213, 28, 255, 137, 194, 180, 108, 108, 
  108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 
  108, 108, 108, 108, 108, 108, 108, 108
};
 //current note on's for voices A, B, and C.  Use note value 0 for "not on". 
byte CurrentNoteOn[3] = {0, 0, 0}; //holds voice note values even during Release
byte VoiceOn[3] = {0, 0, 0};  // reflects voice off and on from MIDI NoteON/Off

byte Mode[3] = { 0, 0, 0 };  // sets up volume by 4-bit Amp, or Envelopes
 
//____________________________________________________
//                      SETUP()
//____________________________________________________


void setup() {

delay(1000);

//MIDI Callback Handle references here
//MIDI.begin(MIDI_CHANNEL_OMNI);

  MIDI.setHandleNoteOn(myHandleNoteOn); //for Callback MIDI In 
  MIDI.setHandleNoteOff(myHandleNoteOff); //for Callback MIDI In 
  MIDI.begin(MIDI_CHANNEL_OMNI);

pinMode(LED, OUTPUT);
digitalWrite(LED, HIGH);


pinMode(SIDRW, OUTPUT);
digitalWrite(SIDRW, LOW);    // High for Read, Low for Write
pinMode(SIDCS, OUTPUT);
digitalWrite(SIDCS, HIGH); //Chip Select active low,Read or Write to register

pinMode(VoiceD1, OUTPUT);
digitalWrite(VoiceD1, LOW);
pinMode(VoiceD2, OUTPUT);
digitalWrite(VoiceD2, LOW);
pinMode(VoiceD3, OUTPUT);
digitalWrite(VoiceD3, LOW);

pinMode(Switch1, INPUT);  // Set up switch inputs with pullup resistors
digitalWrite(Switch1, HIGH);
pinMode(Switch2, INPUT);
digitalWrite(Switch2, HIGH);
pinMode(Switch3, INPUT);
digitalWrite(Switch3, HIGH);
pinMode(Switch4, INPUT);
digitalWrite(Switch4, HIGH);

DDRA = B11111111;  // outputs
DDRB = B11111111;
DDRC = B11111111;
                    
resetSID();

// ~~~~~~~~~~~~~~Setup Synth values~~~~~~~~~~~~~~~~~~~~

Attack[1] = 0;
Decay[1] = 0;
ldEnvAD(1);
Sustainx[1] = 15;
Release[1] = 0;
ldEnvSR(1);
Waveshape[1] = TRIANGLE;

Attack[2] = 0;
Decay[2] = 0;
ldEnvAD(2);
Sustainx[2] = 15;
Release[2] = 0;
ldEnvSR(2);
Waveshape[2] = TRIANGLE;

Attack[3] = 0;
Decay[3] = 0;
ldEnvAD(3);
Sustainx[3] = 15;
Release[3] = 0;
ldEnvSR(3);
Waveshape[3] = TRIANGLE;

ldFiltRes(8, 0);
ldModeVol(OFF3, 15);

                   FreqHi[1]=SIDMidiNoteHi[50]; 
                   ldFreqHi(1);
                   FreqLo[1]=SIDMidiNoteLow[50];
                   ldFreqLo(1);              
                   ldGate(1, 1);  
                   delay(1000);
                   FreqHi[2]=SIDMidiNoteHi[60]; 
                   ldFreqHi(2);
                   FreqLo[2]=SIDMidiNoteLow[60];
                   ldFreqLo(2);
                   ldGate(2, 1);    
                   delay(2000);
                    ldGate(1, 0);  
                    ldGate(2, 0);  

Serial.begin(9600);

digitalWrite(LED, HIGH);

//____________________________________________________

} //End of Setup

//____________________________________________________
//                      MAIN LOOP
//____________________________________________________

void loop() {

  digitalWrite(LED, HIGH);
 
  loadSensors();
  readRegisters();
  tone(VoiceD1, (frontPot << 1));  // set frequency of Voice D

// Frequency offsets for Voices 1 and 2 from Sliders 1 and 2

  FreqHi[1] = SIDMidiNoteHi[CurrentNoteOn[0]] + (slider1 >> 3) ;
  ldFreqHi(1);
  FreqHi[2] = SIDMidiNoteHi[CurrentNoteOn[1]] + (slider2 >> 3) ;
  ldFreqHi(2);

// Course and Fine tune of Modulating Voice 3 from Sliders 3 and 6

  FreqLo[3]=slider5 >> 2;
  ldFreqLo(3);
  FreqHi[3] = (slider3 >> 2) + (slider5 >> 8);
  ldFreqHi(3);

// Pulse width of Voices 1 and 2 (12 bits) from Sliders 4 and 5 (10 bits)

  PulseWHi[1] = slider4 >> 6;
  ldPulseWHi(1);
  PulseWLo[1] = slider4 << 2;
  ldPulseWLo(1);

  PulseWHi[2] = slider5 >> 6;
  ldPulseWHi(2);
  PulseWLo[2] = slider5 << 2;
  ldPulseWLo(2);

  // Filter cutoff Frequency (11 bits, ignore lower 3) from potX (8 bits)
  ldFCHi(potX);

 //SoftPot used for loading Voice Waveforms, Envelopes, Sync/RingMod modes, Filter Modes
 
   if (softPot > 15 && softPot < 200) {  //tap bottom of pot
   // Serial.print(softPot);
   // Serial.println(" bottom");
     Waveshape[1] = 0;
     if (switch1){
     Waveshape[1]=(PULSE * !switch2)+(SAWTOOTH * !switch3)+(TRIANGLE * !switch4);}                 
     else { Waveshape[1] = NOISE; } 
          if (slider5 > 20) {Waveshape[1] = TRIANGLE + RINGMOD;} 
          if (slider6 > 20) {Waveshape[1] = Waveshape[1] + SYNC;}          
          
          Attack[1] = slider1 >> 6;
          Decay[1] = slider2 >> 6;
          Sustainx[1] = slider3 >> 6;
          Release[1] = slider4 >> 6;
          ldEnvAD(1);
          ldEnvSR(1);
          delay(1000);
          }
   
       else if ((softPot > 400) && (softPot < 600)){  //tap middle of pot
        //Serial.print(softPot);
       // Serial.println( " middle");
          Waveshape[2] = 0;
          if (switch1){
     Waveshape[2]=(PULSE * !switch2)+(SAWTOOTH * !switch3)+(TRIANGLE * !switch4);}                 
     else { Waveshape[2] = NOISE; }           
          if (slider5 > 20) {Waveshape[2] = TRIANGLE + RINGMOD;} 
          if (slider6 > 20) {Waveshape[2] = Waveshape[2] + SYNC;}
           
          Attack[2] = slider1 >> 6;
          Decay[2] = slider2 >> 6;
          Sustainx[2] = slider3 >> 6;
          Release[2] = slider4 >> 6;
          ldEnvAD(2);
          ldEnvSR(2);
          delay(1000);
          }
       
       else if (softPot > 800) {  //tap top of pot
        //Serial.print(softPot);
        //Serial.println(" top");
          
          if (slider1 > 10) {hp=1;} else {hp=0;}
          if (slider2 > 10) {bp=1;} else {bp=0;}
          if (slider3 > 10) {lp=1;} else {lp=0;}

          filt = (!switch1 * FILT1) + (!switch2 * FILT2) + (!switch3 * FILT3) + (!switch4 * FILTEX);
          ldFiltRes(filt, potY >> 4);
          ldModeVol( (hp * HP) + (lp * LP) + (bp * BP) + OFF3, 15);
          delay(1000);
          }


  MIDI.read();  
  
} //End of Main Loop

//____________________________________________________
//                SID Basic Address and Data Functions
//____________________________________________________

void loadAddress(int address){
 PORTB = address;
}

void loadData(int data){ 
  PORTC = data;   
  digitalWrite(SIDCS, LOW);
  delayMicroseconds(3);  
  digitalWrite(SIDCS, HIGH);
}

void resetSID(){
  for(int i=0; i<25; i++){
    loadAddress(i);
    loadData(0);
  } 
  ldModeVol(LP, 255);
  }

void readRegisters(){      //Collect values of all 4 SID readable registers
  DDRC = B00000000;        //Setup Data Lines as Inputs
  digitalWrite(SIDRW, HIGH);  // Setup Read/Write for a Data Read
  
  loadAddress(25);
  digitalWrite(SIDCS, LOW);
  delayMicroseconds(3);
  potX = PINC;
  delayMicroseconds(3);
  digitalWrite(SIDCS, HIGH);
  
  loadAddress(28);
  digitalWrite(SIDCS, LOW);
  delayMicroseconds(3);
  env3 = PINC;
  delayMicroseconds(3);
  digitalWrite(SIDCS, HIGH);

  loadAddress(27);
  digitalWrite(SIDCS, LOW);
  delayMicroseconds(3);
  osc3_rand = PINC;
  delayMicroseconds(3);
  digitalWrite(SIDCS, HIGH);

  loadAddress(26);
  digitalWrite(SIDCS, LOW);
  delayMicroseconds(3);
  potY = PINC;
  delayMicroseconds(3);
  digitalWrite(SIDCS, HIGH);

  digitalWrite(SIDRW, LOW);  // Reset Read/Write to Data Write
  DDRC = B11111111;       // Reset Data Lines as Outputs
}


void loadSensors(){      // load all current sensor values
      slider1 =   analogRead(Slider1);
      slider2 =   analogRead(Slider2);
      slider3 =   analogRead(Slider3);
      slider4 =   analogRead(Slider4);
      slider5 =   analogRead(Slider5);
      slider6 =   analogRead(Slider6);
      softPot =   analogRead(SoftPot);
      frontPot =  analogRead(FrontPot);
      lightSensor1 =   analogRead(LightSensor1) >> 2;
      lightSensor2 =   analogRead(LightSensor2) >> 2;
      lightSensor3 =   analogRead(LightSensor3) >> 2;
      lightSensor4 =   analogRead(LightSensor4) >> 2;
      lightSensor5 =   analogRead(LightSensor5) >> 2;
      lightSensor6 =   analogRead(LightSensor6) >> 2;
      switch1 = digitalRead(Switch1);
      switch2 = digitalRead(Switch2);
      switch3 = digitalRead(Switch3);
      switch4 = digitalRead(Switch4);     
}
//____________________________________________________
//               SID Individual Control Register Load 
//____________________________________________________

  /*       ldFreqLo, ldFreqHi,        --voices 1, 2, 3   Frequency
           ldPulseWLo, ldPulseWHi,    --voices 1, 2, 3   Pulse Width
           ldGate,                    --voices 1, 2, 3   Gate the Envelope (+ Waveshape)
           ldEnvAD, ldEnvSR,          --voices 1, 2, 3   Envelope ADSR
           ldFCLo, ldFCHi, ldResFilt  --Filter Cutoff Frequency/Resonance
           ldModeVol                  --FilterType/OutputVolume
*/

void ldFreqLo(int voice){  //8-bit fine tune frequency -- FreqLo
  loadAddress(addr[voice]);
  loadData(FreqLo[voice] & 255);
}

void ldFreqHi(int voice){  //8-bit course tune frequency -- FreqHi
  loadAddress(addr[voice]+1);
  loadData(FreqHi[voice] & 255);
}

void ldPulseWLo(int voice){  //8-bit fine tune Pulse Width -- PulseWLo
  loadAddress(addr[voice]+2);
  loadData(PulseWLo[voice] & 255);
}

void ldPulseWHi(int voice){  //4-bit course tune Pulse Width -- PulseWHi
  loadAddress(addr[voice]+3);
  loadData(PulseWHi[voice] & 15);
}

void ldGate(int voice, int x){  // Gates the ADSR Envelope (also loads Waveshape)
  
  loadAddress(addr[voice]+4);
  loadData(Waveshape[voice] + x);
}

void ldEnvAD(int voice){  //4-bit Attack Time, 4-bit Decay time -- Env Attack/Decay
  loadAddress(addr[voice]+5);
  int x = ((Attack[voice] & 15) << 4) + (Decay[voice] & 15);
  loadData(x);
}


void ldEnvSR(int voice){  //4-bit Sustain Level, 4-bit Release time -- Env Sustain/Release
  loadAddress(addr[voice]+6);
  int x = ((Sustainx[voice] & 15) << 4) + (Release[voice] & 15);
  loadData(x);
}

void ldFCLo(int data){  //3-bit fine tune Filter Cutoff Frequency
  loadAddress(21);
  loadData(data & 7);
}

void ldFCHi(int data){  //8-bit course tune Filter Cutoff Frequency
  loadAddress(22);
  loadData(data & 255);
}

void ldFiltRes(int filt, int res){   //FILTEX/FILT3/FILT2/FILT1, 4-bit Filter Resonance, 
  loadAddress(23);
  int x = ((res & 15) << 4) + (filt & 15);
  loadData(x);
}

void ldModeVol(int mode, int vol){   //Filter Type 3OFF/HP/BP/LP, 4-bit Output Volume
  loadAddress(24);
  loadData((vol & 15) + (mode & B11110000));
}


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      Callback MIDI_In Handles
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// user created Callback Function for MIDI Input Test

 void myHandleNoteOn(byte channel, byte note, byte velocity){
  
    for (int x = 0; x < 2; x ++){    // search through the 2 SID voices

    if (VoiceOn[x] == 0){      // this SID voice is available 
          if (velocity > 0){
                 switch (x) {           
                 case 0: 
                   FreqHi[1]=SIDMidiNoteHi[note]; 
                   ldFreqHi(1);
                   FreqLo[1]=SIDMidiNoteLow[note];
                   ldFreqLo(1);              
                   ldGate(1, 1);                                                  
                   CurrentNoteOn[x] = note; 
                   VoiceOn[x] = 1;    
                   break;                                 
                case 1:
                   FreqHi[2]=SIDMidiNoteHi[note]; 
                   ldFreqHi(2);
                   FreqLo[2]=SIDMidiNoteLow[note];
                   ldFreqLo(2);
                   ldGate(2, 1);                                      
                   CurrentNoteOn[x] = note;
                   VoiceOn[x] = 1;      
                   break;                                              
                 }  // end of switch
                 
          }  // end of if (velocity > 0)
          
          else if (velocity == 0){
            
                 switch (x) {           
                 case 0:  
                   ldGate(1, 0);                  
                   VoiceOn[x] = 0;     
                   break;                                 
                case 1:
                   ldGate(2, 0);                 
                   VoiceOn[x] = 0;          
                   break;                                 
                
            }  // end of switch for velocity 0
          }  // end of if (velocity == 0)
              
                 break;  //  break out of loop   
                                              
     }  // end of Note On check

     
}  // end of for loop
}  // end of Handle
 


 void myHandleNoteOff(byte channel, byte note, byte velocity){
 for (int x = 0; x < 3; x ++){          // search through the 3 AY voices

    if (CurrentNoteOn[x] == note){      // which voice is playing the note to turn off?

                 switch (x) {           
                 case 0:  
                   ldGate(1, 0);                  
                   VoiceOn[x] = 0;     
                   break;                                 
                case 1:
                   ldGate(2, 0);                 
                   VoiceOn[x] = 0;          
                   break;                                 
                
                 }  // end of switch 
                                                                           
     }  // end of Note check

     
}  // end of for loop
}  //end of Handle





