
/* 
                  
                ADC1 to DAC0 with buffer delay
                
Send samples from a live ADC input to an array buffer.  
Output older samples from the array to DAC1334 using the I2S library.

If Switch1 is up the maximum delay is set up.
If Switch1 is down the delay is set by Slider1.

The MKRZero has only 32K bytes of SRAM memory used to store program variables.
The buffer array is limited to less than this size (12k x 2 bytes).  The array is circular,
an input pointer and an output pointer are incremented after every sample operation and
wrapped back to point to zero on reaching the top of the array.

Thanks to "Arduino Music and Audio Projects" by Mike Cook
*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      CONSTANTS and Variables
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#include "I2S.h"
const int sampleRate = 10000; //sample rate for I2S.h 

#define BUFFER_SIZE 8200  //10k x 2 bytes = 20k bytes, limited by the 32k SRAM size
short buffer1[BUFFER_SIZE]; // the size of a short is 2 bytes (16 bits)

// ANALOG INPUTS
//
#define SLIDER1 A5  //top left
#define SLIDER2 A3  //bottom left
#define SLIDER3 A4  //top right
#define SLIDER4 A2  //bottom right

//
//DIGIITAL SWITCHES
//
#define SWITCH1 5  //top toggle switch
#define SWITCH2 4  //bottom toggle switch
#define SWITCH3 6  //right pushbutton switch
#define SWITCH4 7  //left pushbuttun switch

boolean switch1 = 0;
boolean switch2 = 0;
boolean switch3 = 0;
boolean switch4 = 0;

#define VOICEPIN_A 8   //pot 4 on box right side
#define VOICEPIN_B 9   //pot 5 on box right side
#define VOICEPIN_C 10  //Modulating Voice (white knob on box left side, and switch up)

#define LED1 1  //top toggle switch's LED
#define LED2 0  //bottom toggle switch's LED

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      SETUP()
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void setup() {

// Initialize the buffer contents to all zero
  for (int i=0; i<BUFFER_SIZE; i++){
    buffer1[i] = 0;
  }

//   Serial.begin(9600);  //used only for printing samplerate times

pinMode(LED1, OUTPUT);
digitalWrite(LED1, HIGH);
pinMode(LED2, OUTPUT);
digitalWrite(LED2, LOW);


pinMode(VOICEPIN_A, OUTPUT);
digitalWrite(VOICEPIN_A, LOW);
pinMode(VOICEPIN_B, OUTPUT);
digitalWrite(VOICEPIN_B, LOW);
pinMode(VOICEPIN_C, OUTPUT);
digitalWrite(VOICEPIN_C, LOW);

pinMode(SWITCH1, INPUT); //Switch inputs have external 10k pulldown resistor
pinMode(SWITCH2, INPUT);
pinMode(SWITCH3, INPUT);
pinMode(SWITCH4, INPUT);
 
// Configure serial port.  
//                       Serial.begin(500000);
 //                      while (!Serial) { delay(10); } //waits for you to open Serial Monitor
 //                      Serial.println("Zero I2S Audio Tone Generator");

// start I2S at the sample rate with 16-bits per sample
  if (!I2S.begin(I2S_PHILIPS_MODE, sampleRate, 16)) {
    Serial.println("Failed to initialize I2S");
    while (1); //do nothing if failed
  }

// set up speed of ADCs

ADC->CTRLB.reg = ADC_CTRLB_PRESCALER_DIV32 |  //Divde 48Khz GCLK by 32 for ADC
                 ADC_CTRLB_RESSEL_12BIT;       //Set ADC resolution to 12 bits
while(ADC->STATUS.bit.SYNCBUSY);              // Wait for these changes to sync

//Sampling Time Length SAMPLEN (normally 63) allows time for ADC capacitor to charge
ADC->SAMPCTRL.reg = ADC_SAMPCTRL_SAMPLEN(16);  //Set Sampling Time Length to 16

analogReadResolution(12);  //sample manipulations are done at higher resolution
analogWriteResolution(10); //highest resolution of MKRZero DAC

}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                     Main LOOP
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loop() {  

static signed int bufferIn = BUFFER_SIZE - 1; //input pointer set to array top
static unsigned int bufferOffset = BUFFER_SIZE - 1;   //output offset from input, sets maximum delay 
static signed int bufferOut = BUFFER_SIZE - bufferOffset; //output pointer
static signed int sample; 
int timeIn;
 
while(1){  // inner loop for higher sample rate

 timeIn = micros();

 //set up different delay times
  if (digitalRead(SWITCH1)){    
    bufferOffset = BUFFER_SIZE - (analogRead(SLIDER1) << 1 ); // variable delay
  }
    else {
    bufferOffset = BUFFER_SIZE - 10;  // maximum delay
  }
  
  sample = analogRead(A1) - 2048;  //make audio bipolar, plus and minus with 12 bit resolution
  //any multiply or divide operations must be done on bipolar signals with zero offsets

  //do any operations on the samples here
  
  buffer1[bufferIn] = sample;  //save sample in delay array

  sample = sample + buffer1[bufferOut] << 3 ; //add original plus delayed 
   
//                       Serial.println(sample);

  I2S.write(sample); // write twice for left and right outputs
  I2S.write(sample);

// update the two circular array pointers, input and output

 bufferIn++;
  if (bufferIn >= BUFFER_SIZE){ bufferIn = 0; }
  
  bufferOut++;
  if (bufferOut >= BUFFER_SIZE) {    
      bufferOut = bufferIn - bufferOffset ;
      if (bufferOut < 0){ bufferOut += BUFFER_SIZE;}
  }

//timeIn = micros() - timeIn;
//Serial.println(timeIn);  //print sample rate time in microseconds
 
} //end of while 
} //End of loop


//````````````````````````````````````````````````````````````
