
// Connect an I2S DAC or amp (like the UDA1334A) to the Arduino Zero
// and play back from ADC A1 with signal processing using the I2S.h library.
/*
An input/output transfer function is setup for both the positive and negative going parts of the signal.

Each possible input sample value, -2048 to 0 to +2048, from the 12-bit A1 ADC  is translated into 
a different output sample value.  This can be accomplished with a 4096 cell table, or calculated with a 
function, as done here.

The xy (input|output) function is made up of two straight line functions.  The first is simple - input equals
output -- no change to input values zero to plus and minus a value determined by Slider1.  The second straight 
line starts at the Slider1 value and ends at an output value determined by Slider3 and results in a distortion
of the tops and bottoms of the input waveform.  

The two straigt line functions for positive input values zero to 2048 are mirrored in the negative input values zero 
to -2048.  

If Slider3 gives a value of 2048, then input equals output for all possible input values.  If Slider2 equals
Slider1, then output waveforms are chopped off at that value.  If Slider2 is between 2048 and Slider1, then the 
output waveform tops and bottoms are a squished version of the input.  If Slider2 is less than Slider1, the waveform
tops and bottoms are inverted from their input shapes.

Slider 1 and 3 values change only when Switch1 is down.

Slider2 pans between the input waveform and the distorted output waveform.
The SampleRate can be raised to 16000 if this slow pan calculation is commented out.
 
*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#include "I2S.h"
const int sampleRate = 10000; //sample rate for I2S.h 

signed int a = 0;
signed int b = 0;
signed int sample = 0;
signed int samplex = 0;
float pan = 0;

// ANALOG INPUTS
//
#define SLIDER1 A5  //top left
#define SLIDER2 A3  //bottom left
#define SLIDER3 A4  //top right
#define SLIDER4 A2  //bottom right

//
//DIGIITAL SWITCHES
//
#define SWITCH1 5  //top toggle switch
#define SWITCH2 4  //bottom toggle switch
#define SWITCH3 6  //right pushbutton switch
#define SWITCH4 7  //left pushbuttun switch

boolean switch1 = 0;
boolean switch2 = 0;
boolean switch3 = 0;
boolean switch4 = 0;

#define VOICEPIN_A 8   //pot 4 on box right side
#define VOICEPIN_B 9   //pot 5 on box right side
#define VOICEPIN_C 10  //Modulating Voice (white knob on box left side, and switch up)

#define LED1 1  //top toggle switch's LED
#define LED2 0  //bottom toggle switch's LED

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void setup() {
  
// Configure serial port.  
//                       Serial.begin(500000);

pinMode(LED1, OUTPUT);
digitalWrite(LED1, HIGH);
pinMode(LED2, OUTPUT);
digitalWrite(LED2, HIGH);

pinMode(VOICEPIN_A, OUTPUT);
digitalWrite(VOICEPIN_A, LOW);
pinMode(VOICEPIN_B, OUTPUT);
digitalWrite(VOICEPIN_B, LOW);
pinMode(VOICEPIN_C, OUTPUT);
digitalWrite(VOICEPIN_C, LOW);

pinMode(SWITCH1, INPUT); //Switch inputs have external 10k pulldown resistor
pinMode(SWITCH2, INPUT);
pinMode(SWITCH3, INPUT);
pinMode(SWITCH4, INPUT);


// start I2S at the sample rate with 16-bits per sample
  if (!I2S.begin(I2S_PHILIPS_MODE, sampleRate, 16)) {
    Serial.println("Failed to initialize I2S");
    while (1); //do nothing if failed
  }

// set up speed of ADCs

  ADC->CTRLB.reg = ADC_CTRLB_PRESCALER_DIV32 |  //Divde 48Khz GCLK by 32 for ADC
                 ADC_CTRLB_RESSEL_12BIT;       //Set ADC resolution to 12 bits
  while(ADC->STATUS.bit.SYNCBUSY);              // Wait for these changes to sync

//Sampling Time Length SAMPLEN (normally 63) allows time for ADC capacitor to charge
  ADC->SAMPCTRL.reg = ADC_SAMPCTRL_SAMPLEN(1);  //Set Sampling Time Length to 1
  
}   //End setup

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loop() {

  samplex = analogRead(A1) - 2048;
  sample = samplex;  //keep samplex as the original input signal
 
  
  if (digitalRead(SWITCH1)){
  a = analogRead(SLIDER1)  >> 2;
  b = analogRead(SLIDER3) >> 2 ;
  }

  if (sample >= 0) {  //for positive samples
    
      if (sample > a){                    // if sample < a, leave it unchanged    
        sample = a + (  ((b - a) * (sample - a)) / (2048 - a)  ) ;  //transfer function
      }
  
  
  } //End positive samples

  
  else {  //for negative samples

       sample = -sample;
       if (sample > a){                    // if sample < a, leave it unchanged    
        sample = a + (  ((b - a) * (sample - a)) / (2048 - a)  ) ;  //transfer function
      }
      sample = -sample;
    
  }  //End negative samples


  pan = analogRead(SLIDER2) ;
  pan = pan / 4095;  //pan is a float between 0 and 1
  
sample = (pan * sample) + ((1 - pan) * samplex);  //pan between input and distorted version

//              Connect a sinewave to the A1 input and use the Serial Plotter to see the output waveform
//                           Serial.println(sample); //use the Serial Plotter to 
 
 sample = sample << 4;  //12-bit to 16-bit size
              

  I2S.write(sample);  // write twice for left and right outputs
  I2S.write(sample);

}   //End loop

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
