
/* 
     MIDI 3-voice random notes

  For each of 3 voices, the sliders can change Frequency Base, Frequency Range,
  Note Amplitude and Note Duration.  Since there are only 4 sliders, Slider 4
  is used for Note Duration, and sliders 1, 2, and 3 are shared between the 3 voices
  to affect Base, Range, and Amplitude.
  
  Two toggle switches determine which of the 3 voices is affected by the first 3 sliders.
  
  A Slider position is divided into 5 ranges. Each of the 5 slider ranges is assigned an
  increment value of +2, +1, 0, -1, or -2.  Each time through the main program loop
  the Note parameter is incremented (or decremented) by the value assigned to that slider position.

  Sliders set in the middle have an increment value of zero, resulting in no
  change to any note parameters. 
  
  Uses Serial1 for MIDI, so will work on any board
  with 2 hardware serial ports: MKR boards, Leonardo, Micro, or Yún

  On the MKR Zero TX is D14, RX is D13

  Circuit:
    connect TX of Serial1 to pin5 of MIDI jack
    and connect pin4 of MIDI jack to a 220 ohm resistor to 3.3v power
*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      CONSTANTS and Variables
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// 
// ANALOG INPUTS
//
#define SLIDER1 A5  //top left
#define SLIDER2 A3  //bottom left
#define SLIDER3 A4  //top right
#define SLIDER4 A2  //bottom right

int slider1 = 0;
int slider2 = 0;
int slider3 = 0;
int slider4 = 0;

//
//DIGIITAL SWITCHES
//
#define SWITCH1 5  //top toggle switch
#define SWITCH2 4  //bottom toggle switch
#define SWITCH3 6  //right pushbutton switch
#define SWITCH4 7  //left pushbuttun switch

boolean switch1 = 0;
boolean switch2 = 0;
boolean switch3 = 0;
boolean switch4 = 0;
int swtchCombo = 0;

#define VOICEPIN_A 8   //pot 4 on box right side
#define VOICEPIN_B 9   //pot 5 on box right side
#define VOICEPIN_C 10  //Modulating Voice (white knob on box left side, and switch up)

#define LED1 1  //top toggle switch's LED
#define LED2 0  //bottom toggle switch's LED


//
//  3-VOICE VARIABLES
//

int dur = 200;
int durA = 0;
int durB = 0;
int durC = 0;

int baseA = 0;
int baseB = 0;
int baseC = 0;

int baseAincrement = 0;
int baseBincrement = 0;
int baseCincrement = 0;

int rangeA = 0;
int rangeB = 0;
int rangeC = 0;

int rangeAincrement = 0;
int rangeBincrement = 0;
int rangeCincrement = 0;

int freqA = 50;
int freqB = 75;
int freqC = 100;

int ampA = 0;
int ampB = 0;
int ampC = 0;

int ampAincrement = 0;
int ampBincrement = 0;
int ampCincrement = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                      SETUP()
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void setup() {

  delay(1000);

  // initialize MIDI serial:
  Serial1.begin(31250);

pinMode(LED1, OUTPUT);
digitalWrite(LED1, HIGH);
pinMode(LED2, OUTPUT);
digitalWrite(LED2, HIGH);


pinMode(VOICEPIN_A, OUTPUT);
digitalWrite(VOICEPIN_A, LOW);
pinMode(VOICEPIN_B, OUTPUT);
digitalWrite(VOICEPIN_B, LOW);
pinMode(VOICEPIN_C, OUTPUT);
digitalWrite(VOICEPIN_C, LOW);

pinMode(SWITCH1, INPUT); //Switch inputs have external 10k pulldown resistor
pinMode(SWITCH2, INPUT);
pinMode(SWITCH3, INPUT);
pinMode(SWITCH4, INPUT);

}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                     Main LOOP
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loop() {

  delay(1);

  loadSensors();

  zeroIncrements();

  swtchCombo = switchCombo();

  switch (swtchCombo){  //sliders affect voice set by switches
    case 1:
            baseAincrement = sliderIncrement(slider1);
            rangeAincrement = sliderIncrement(slider2);
            ampAincrement = sliderIncrement(slider3);
    break;
    
    case 2:
            baseBincrement = sliderIncrement(slider1);
            rangeBincrement = sliderIncrement(slider2);
            ampBincrement = sliderIncrement(slider3);
    break;
    
    case 3:
            baseCincrement = sliderIncrement(slider1);
            rangeCincrement = sliderIncrement(slider2);
            ampCincrement = sliderIncrement(slider3);
    break;
    
    default:
    break;
    
  }
  

// ------------Voice A ---------------------

     if (durA > 0){  --durA;  }   //wait for a count of durA
                                        
     else{  // when envelope reaches zero, reset voice A with new frequency and envelope

            midiCommand(0x90, freqA, 0);  //turn off note
            
            baseA = baseA + baseAincrement;  //get new pitch base for voice A
            baseA = constrain(baseA, 20, 100);
            
            rangeA = rangeA + rangeAincrement; //get new pitch range for voice A
            rangeA = constrain(rangeA, 0, 50);
            
            freqA = freqA + random(rangeA);  //calculate new pitch for voice A
            freqA = constrain(freqA, 20, 100); //MIDI note constraints

            ampA = ampA + ampAincrement;  //get amp for voice A
            ampA = constrain(baseA, 0, 127); //MIDI note velocity constraints

            durA = random(200, 200 + slider4) ;  // get random duration
            
            midiCommand(0x90, freqA, ampA);  //turn on new note
   }
                            
             

 // ------------Voice B ---------------------

     if (durB > 0){  --durB;  }   //wait for a count of durB
                                        
     else{  // when envelope reaches zero, reset voice A with new frequency and envelope

            midiCommand(0x90, freqB, 0);  //turn off note
            
            baseB = baseB + baseBincrement;  //get new pitch base for voice B
            baseB = constrain(baseB, 20, 100);
            
            rangeB = rangeB + rangeBincrement; //get new pitch range for voice B
            rangeB = constrain(rangeB, 0, 50);
            
            freqB = freqB + random(rangeB);  //calculate new pitch for voice B
            freqB = constrain(freqB, 20, 100); //MIDI note constraints

            ampB = ampB + ampBincrement;  //get amp for voice B
            ampB = constrain(baseB, 0, 127); //MIDI note velocity constraints

            durB = random(200, 200 + slider4) ;  // get random duration
            
            midiCommand(0x90, freqB, ampB);  //turn on new note
   }
                 

// ------------Voice C ---------------------

     if (durC > 0){  --durC;  }   //wait for a count of durC
                                        
     else{  // when envelope reaches zero, reset voice A with new frequency and envelope

            midiCommand(0x90, freqC, 0);  //turn off note
            
            baseC = baseC + baseCincrement;  //get new pitch base for voice C
            baseC = constrain(baseC, 20, 100);
            
            rangeC = rangeC + rangeCincrement; //get new pitch range for voice C
            rangeC = constrain(rangeC, 0, 50);
            
            freqC = freqC + random(rangeC);  //calculate new pitch for voice C
            freqC = constrain(freqC, 20, 100); //MIDI note constraints

            ampC = ampC + ampCincrement;  //get amp for voice A
            ampC = constrain(baseC, 0, 127); //MIDI note velocity constraints

            durC = random(200, 200 + slider4) ;  // get random duration
           
            midiCommand(0x90, freqC, ampC);  //turn on new note
   }
                                     
 // ------------Switch Slows everything to almost a standstill---------------------    
    
                      if (switch3 == 1){ delay(250); }  
} // End of Loop

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                    Functions
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


// send a 3-byte midi message
void midiCommand(byte cmd, byte data1, byte  data2) {
  Serial1.write(cmd);     // command byte (should be > 127)
  Serial1.write(data1);   // data byte 1 (should be < 128)
  Serial1.write(data2);   // data byte 2 (should be < 128)
}

void loadSensors(){      // load all current sensor values
      slider1 =   analogRead(SLIDER1) ;
      slider2 =   analogRead(SLIDER2) ;
      slider3 =   analogRead(SLIDER3) ;
      slider4 =   analogRead(SLIDER4) ;
      
      switch1 = digitalRead(SWITCH1);
      switch2 = digitalRead(SWITCH2);
      switch3 = digitalRead(SWITCH3);
      switch4 = digitalRead(SWITCH4); 
}

int switchCombo(){   // set LEDs to reflect switch state, switch down = LED on
    int result = switch2 + (switch1 * 2);

    switch (result) {
      case 0:
              digitalWrite(LED2, LOW);
              digitalWrite(LED1, LOW);
              break;
      case 1:
              digitalWrite(LED2, HIGH);
              digitalWrite(LED1, LOW);
              break;
      case 2:
              digitalWrite(LED2, LOW);
              digitalWrite(LED1, HIGH);
              break;
      case 3:
              digitalWrite(LED2, HIGH);
              digitalWrite(LED1, HIGH);
              break;
    }
    return result;   
}

int sliderIncrement(int slider){  //slider position determines incremental changes
    int result;
    
    if((slider <= 600) && (slider > 400)){ result = 0; } //do nothing middle state
    else if((slider > 800)){ result = 2; }  // double increase
    else if((slider <= 200)){ result = -2; }  // double decrease
    else if((slider <= 400) && (slider > 200)){ result = -1; }  //single decrease
    else { result = 1; }  //single increase  (range between 600 and 800)

    return result;
}

void zeroIncrements(){   //set to no incremental changes in freq. base, range and amp
  
            baseAincrement = 0;
            rangeAincrement = 0;
            ampAincrement = 0;  

            baseBincrement = 0;
            rangeBincrement = 0;
            ampBincrement = 0;  

            baseCincrement = 0;
            rangeCincrement = 0;
            ampCincrement = 0;              
}
