/*
 * 
 *  3-VOICE ARDUINO SYNTHESIZER USING TIMER INTERRUPTS
      
      3 squarewave tones produced from Arduino pins 
      
      Slider3 - Sets Frequency of Tone3 with arduino tone(), used as the modulating voice
      Slider1 - Sets Frequency of Tone1
      Slider2 - Sets Frequency of Tone2

      Switch1 - turns off or on Tone1 slider adjustments
      Switch2 - turns off or on Tone2 slider adjustments
      Switch4 - Loads slider value for modulating voice

Voices are created from a master timer interrupt.

The interrupt function is run at regular intervals as set by a very high speed timer.
The timed interrupt function decrements freq values and toggles digital pin outputs
when they reach zero.  Freq values determine the pitch of the voices and are varied from sliders. 

This sketch illustrates how to set up a timer on an Atmel SAMD21 based Arduino board (MKR Zero)
based on the ARM Cortex M0+.  The Arduino IDE includes the CMSIS
library for working with the SAMD21 registers directly.  Here it is used to set up timers and interrupts. 

Thanks to:
Timer5 library for Arduino Zero and MKR1000 
  (only for SAMD arch.)
  Copyright (c) 2016 Michael Blank, OpenSX. All right reserved.

  based on the code of the AudioZero library by:
  Arturo Guadalupi <a.guadalupi@arduino.cc>
  Angelo Scialabba <a.scialabba@arduino.cc>
  Claudio Indellicati <c.indellicati@arduino.cc> <bitron.it@gmail.com>  
*/

uint32_t sampleRate = 20000; //sample rate, determines how often TC5_Handler is called
//
// ANALOG INPUTS
//
#define SLIDER1 A5
#define SLIDER2 A3
#define SLIDER3 A4
#define SLIDER4 A2


int slider1 = 0;
int slider2 = 0;
int slider3 = 0;
int slider4 = 0;

//
//DIGIITAL SWITCHES
//
#define SWITCH1 5
#define SWITCH2 4
#define SWITCH3 6
#define SWITCH4 7

boolean switch1 = 0;
boolean switch2 = 0;
boolean switch3 = 0;
boolean switch4 = 0;

#define VOICEPIN_A 8
#define VOICEPIN_B 9
#define VOICEPIN_C 10

#define LED1 1
#define LED2 0

volatile int freq1 = 50;
volatile int freq2 = 50;
volatile int freq3 = 50;

volatile int freq1save = 50;
volatile int freq2save = 50;
volatile int freq3save = 50;
unsigned int load = 50;

unsigned int swtch = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void setup() {

delay(1000);
  
pinMode(LED1, OUTPUT);    //turn on LED1 as power indicator
digitalWrite(LED1, HIGH);

pinMode(LED2, OUTPUT);
digitalWrite(LED2, LOW);


pinMode(VOICEPIN_A, OUTPUT);
digitalWrite(VOICEPIN_A, LOW);
pinMode(VOICEPIN_B, OUTPUT);
digitalWrite(VOICEPIN_B, LOW);
pinMode(VOICEPIN_C, OUTPUT);
digitalWrite(VOICEPIN_C, LOW);

pinMode(SWITCH1, INPUT);  // Set up switch inputs with pullup resistor
pinMode(SWITCH2, INPUT);
pinMode(SWITCH3, INPUT);
pinMode(SWITCH4, INPUT);

  tcConfigure(sampleRate); //configure the timer to run at <sampleRate>Hertz
  
  tcStartCounter(); //starts the timer
   
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void loop() {
  
  //tcDisable(); //This function can be used anywhere if you need to stop/pause the timer
  //tcReset(); //This function should be called everytime you stop the timer

 --load;  //Slider readings are limited since they cause noise in the higher voice frequencies.
 
 if (load <= 0){

  ++swtch;
  if (swtch >=4){ swtch = 1; }

      switch (swtch) {
  case 1:
    if (digitalRead(SWITCH1)){
        freq2save = 2 + (analogRead(SLIDER1) >> 1);
      }
    break;
  case 2:
    if (digitalRead(SWITCH2)){
        freq1save =  2 + (analogRead(SLIDER2) >> 2); 
      }
    break;
  case 3:
    if (digitalRead(SWITCH4)){ 
        freq3save =  2 + (analogRead(SLIDER3) >> 1); 
      }
    break;
}
              
     load = 300;
  }
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

//this function gets called by the interrupt at a high <sampleRate>Hertz
// "TC5_Handler" is in the CMSIS library, doesn't need to be declared as an interrupt handler.

void TC5_Handler (void) {
  
  --freq1;                          //toggle voicePinA at end of freq1 countdown
    if (freq1 <= 0){
        digitalWrite(VOICEPIN_A, !digitalRead(VOICEPIN_A));        
        freq1 = freq1save;
    }
       
     --freq2;                         //toggle voicePinB at end of freq2 countdown
    if (freq2 <= 0){
        digitalWrite(VOICEPIN_B, !digitalRead(VOICEPIN_B));
        freq2 = freq2save;
    }
    
     --freq3; 
      if (freq3 <= 0){
        digitalWrite(VOICEPIN_C, !digitalRead(VOICEPIN_C));
        freq3 = freq3save;  
    }

  TC5->COUNT16.INTFLAG.bit.MC0 = 1; 
  //Writing a 1 to INTFLAG.bit.MC0 clears the interrupt so that it will run again
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/* 
 *  TIMER SPECIFIC FUNCTIONS FOLLOW
 *  you shouldn't change these unless you know what you're doing
 */

//Configures the TC to generate output events at the sample frequency.
//Configures the TC in Frequency Generation mode, with an event output once
//each time the audio sample frequency period expires.

 void tcConfigure(int sampleRate)
{
 // Enable GCLK for TCC2 and TC5 (timer counter input clock)
 //drive the timer from General Clock 0 (CPU clock) and enable the peripheral clock.

        // GCLK1 is 32kHz while GCLK0 is 48MHz 



 
 GCLK->CLKCTRL.reg = (uint16_t) ( GCLK_CLKCTRL_CLKEN |              // Enable clock
                                  GCLK_CLKCTRL_GEN_GCLK0 |          // Select GCLK0 (is 48MHz)
                                  GCLK_CLKCTRL_ID(GCM_TC4_TC5)) ;   // Feed GCLK3 to TC4 and TC5                       
 while (GCLK->STATUS.bit.SYNCBUSY);                                 // Wait for synchronization 
 //accounts for the need to synchronize between different clocks in the chip before using the changed value.




 tcReset();   //resets the timer TC5 settings.



 
 TC5->COUNT16.CTRLA.reg |= TC_CTRLA_MODE_COUNT16;    // Set TC5 Timer counter to be in 16 bit mode and generate a frequency.
 TC5->COUNT16.CTRLA.reg |= TC_CTRLA_WAVEGEN_MFRQ;    // Set TC5 mode as match frequency
 TC5->COUNT16.CTRLA.reg |= TC_CTRLA_PRESCALER_DIV1 | // Set TC5 prescaler to 1 (can be any power of 2) 
                           TC_CTRLA_ENABLE;          // Set TC5 clock enable
 TC5->COUNT16.CC[0].reg  = (uint16_t) (SystemCoreClock / sampleRate - 1); 
 while (tcIsSyncing());                              // Wait for TC5 synchronization
 
 //set TC5 timer counter based off of the system clock (48MHZ) and the user defined sample rate or waveform
 // higher sampleRate value results in more frequent interrupts.

 
 
 
 // Configure interrupt request
 NVIC_DisableIRQ(TC5_IRQn);
 NVIC_ClearPendingIRQ(TC5_IRQn);
 NVIC_SetPriority(TC5_IRQn, 0);
 NVIC_EnableIRQ(TC5_IRQn);

 // Enable the TC5 interrupt request
 TC5->COUNT16.INTENSET.bit.MC0 = 1;
 
 while (tcIsSyncing()); //wait until TC5 is done syncing 
}


 

//Function that is used to check if TC5 is done syncing
//returns true when it is done syncing
bool tcIsSyncing()
{
  return TC5->COUNT16.STATUS.reg & TC_STATUS_SYNCBUSY;
}




//This function enables TC5 and waits for it to be ready
void tcStartCounter()
{
  TC5->COUNT16.CTRLA.reg |= TC_CTRLA_ENABLE; //set the CTRLA register
  while (tcIsSyncing()); //wait until snyc'd
}





//Reset TC5 
void tcReset()
{
  TC5->COUNT16.CTRLA.reg = TC_CTRLA_SWRST;
  while (tcIsSyncing());
  while (TC5->COUNT16.CTRLA.bit.SWRST);
}




//disable TC5
void tcDisable()
{
  TC5->COUNT16.CTRLA.reg &= ~TC_CTRLA_ENABLE;
  while (tcIsSyncing());
}
