#include "Codec.h"
#include <Arduino.h>
#include <Wire.h>

//FOR THE ES8388 CODEC

/* Codec register */
#define Codec_CONTROL1 0x00
#define Codec_CONTROL2 0x01
#define Codec_CHIPPOWER 0x02
#define Codec_ADCPOWER 0x03
#define Codec_DACPOWER 0x04
#define Codec_CHIPLOPOW1 0x05
#define Codec_CHIPLOPOW2 0x06
#define Codec_ANAVOLMANAG 0x07
#define Codec_MASTERMODE 0x08
/* ADC */
#define Codec_ADCCONTROL1      0x09
#define Codec_ADCCONTROL2      0x0a
#define Codec_ADCCONTROL3      0x0b
#define Codec_ADCCONTROL4      0x0c
#define Codec_ADCCONTROL5      0x0d
#define Codec_ADCCONTROL6      0x0e
#define Codec_ADCCONTROL7      0x0f
#define Codec_ADCCONTROL8      0x10
#define Codec_ADCCONTROL9      0x11
#define Codec_ADCCONTROL10     0x12
#define Codec_ADCCONTROL11     0x13
#define Codec_ADCCONTROL12     0x14
#define Codec_ADCCONTROL13     0x15
#define Codec_ADCCONTROL14     0x16
/* DAC */
#define Codec_DACCONTROL1 0x17
#define Codec_DACCONTROL2 0x18
#define Codec_DACCONTROL3 0x19
#define Codec_DACCONTROL4 0x1a
#define Codec_DACCONTROL5 0x1b
#define Codec_DACCONTROL6 0x1c
#define Codec_DACCONTROL7 0x1d
#define Codec_DACCONTROL8 0x1e
#define Codec_DACCONTROL9 0x1f
#define Codec_DACCONTROL10 0x20
#define Codec_DACCONTROL11 0x21
#define Codec_DACCONTROL12 0x22
#define Codec_DACCONTROL13 0x23
#define Codec_DACCONTROL14 0x24
#define Codec_DACCONTROL15 0x25
#define Codec_DACCONTROL16 0x26
#define Codec_DACCONTROL17 0x27
#define Codec_DACCONTROL18 0x28
#define Codec_DACCONTROL19 0x29
#define Codec_DACCONTROL20 0x2a
#define Codec_DACCONTROL21 0x2b
#define Codec_DACCONTROL22 0x2c
#define Codec_DACCONTROL23 0x2d
#define Codec_DACCONTROL24 0x2e
#define Codec_DACCONTROL25 0x2f
#define Codec_DACCONTROL26 0x30
#define Codec_DACCONTROL27 0x31
#define Codec_DACCONTROL28 0x32
#define Codec_DACCONTROL29 0x33
#define Codec_DACCONTROL30 0x34


enum 
{
		SAMPLE_RATE_8000	= 0x0000,
		SAMPLE_RATE_11052	= 0x1000,
		SAMPLE_RATE_12000	= 0x2000,
		SAMPLE_RATE_16000	= 0x3000,
		SAMPLE_RATE_22050	= 0x4000,
		SAMPLE_RATE_24000	= 0x5000,
		SAMPLE_RATE_32000	= 0x6000,
		SAMPLE_RATE_44100	= 0x7000,
		SAMPLE_RATE_48000	= 0x8000,
		SAMPLE_RATE_96000	= 0x9000,
		SAMPLE_RATE_192000	= 0xa000,
};

enum { MODE_MASTER = 0x00,	MODE_SLAVE = 0x01 };

enum 
{
	DATA_FORMAT_I2S		= 0x00,
	DATA_FORMAT_LEFT	= 0x01,
	DATA_FORMAT_RIGHT	= 0x02,
	DATA_FORMAT_DSP		= 0x03,
};

enum 
{
	BCLK_DIV_1			= 0x0,
	BCLK_DIV_2			= 0x1,
	BCLK_DIV_4			= 0x2,
	BCLK_DIV_6			= 0x3,
	BCLK_DIV_8			= 0x4,
	BCLK_DIV_12			= 0x5,
	BCLK_DIV_16			= 0x6,
	BCLK_DIV_24			= 0x7,
	BCLK_DIV_32			= 0x8,
	BCLK_DIV_48			= 0x9,
	BCLK_DIV_64			= 0xa,
	BCLK_DIV_96			= 0xb,
	BCLK_DIV_128		= 0xc,
	BCLK_DIV_192		= 0xd,
};

enum 
{
	LRCK_DIV_16			= 0x0,
	LRCK_DIV_32			= 0x1,
	LRCK_DIV_64			= 0x2,
	LRCK_DIV_128		= 0x3,
	LRCK_DIV_256		= 0x4,
};

Codec::Codec(uint8_t _sda, uint8_t _scl, uint32_t _speed) {
  _pinsda = _sda;
  _pinscl = _scl;
  _i2cspeed = _speed;
  i2c.begin(_sda, _scl, _speed);
}

Codec::~Codec() { i2c.~TwoWire(); }

bool Codec::write_reg(uint8_t reg_add, uint8_t data) {
  i2c.beginTransmission(Codec_ADDR);
  i2c.write(reg_add);
  i2c.write(uint8_t(data & 0xff));
  return i2c.endTransmission() == 0;
}

uint8_t Codec::read_reg(uint8_t reg_add) {
	i2c.beginTransmission(Codec_ADDR);
	i2c.write(reg_add);
	i2c.endTransmission(false);
	uint16_t val = 0u;
	i2c.requestFrom((uint16_t)Codec_ADDR, (uint8_t)1, true);
  if (i2c.available() >= 1)
	{
		val = i2c.read();
	}
	return val;
}; 

bool Codec::identify(int sda, int scl, uint32_t frequency) {
  i2c.begin(sda, scl, frequency);
  i2c.beginTransmission(Codec_ADDR);
  return i2c.endTransmission() == 0;
}

uint8_t* Codec::readAllReg() {
  static uint8_t reg[53];
  for (uint8_t i = 0; i < 53; i++) {
    reg[i] = read_reg(i);
  }
  return reg;
}

bool Codec::init() {
  outCorrectionGain = 1.05924; 
  //correction for -0.5 missmatch of always-on ALC gain
	
  bool res = true;
  /* INITIALIZATION (BASED ON Codec ES8388 USER GUIDE EXAMPLE) */
  // Set Chip to Slave
  res &= write_reg(Codec_MASTERMODE, 0x00);
  // Power down DEM and STM
  res &= write_reg(Codec_CHIPPOWER, 0xFF);
  // Set same LRCK	Set same LRCK
  res &= write_reg(Codec_DACCONTROL21, 0x80);
  // Set Chip to Play&Record Mode
  res &= write_reg(Codec_CONTROL1, 0x05);
  // Power Up Analog and Ibias
  res &= write_reg(Codec_CONTROL2, 0x40);

  /* ADC setting */
  // Micbias for Record
  res &= write_reg(Codec_ADCPOWER, 0x00);

  // Enable Lin1/Rin1 (0x00 0x00) for Lin2/Rin2 (0x50 0x80)
  //res &= writeReg(Codec_ADCCONTROL2,0x00);// LINSEL:LINPUT1, RINSEL:RINPUT1,
	//res &= writeReg(Codec_ADCCONTROL2,0x10);// LINSEL:LINPUT1, RINSEL:RINPUT2,
	//res &= writeReg(Codec_ADCCONTROL2,0x40);// LINSEL:LINPUT2, RINSEL:RINPUT1,
  res &= write_reg(Codec_ADCCONTROL2, 0x50);
  res &= write_reg(Codec_ADCCONTROL3, 0x80);

  // Analog Input PGA gain (0x88 - 24db) (0x77 - 21db)
  res &= write_reg(Codec_ADCCONTROL1, 0x77);

  // SFI setting (i2s mode/16 bit)
  //res &= writeReg(Codec_ADCCONTROL4,0x20); // (I2S – 24Bit)
	//res &= writeReg(Codec_ADCCONTROL4,0x0C); // (I2S – 16Bit, LR )
  res &= write_reg(Codec_ADCCONTROL4, 0x0C);

  //Select MCLK / LRCK ratio for ADC
	//res &= writeReg(Codec_ADCCONTROL5,0x02);// (256, single speed)
	//res &= writeReg(Codec_ADCCONTROL5,0x03);// (384, single speed)
  res &= write_reg(Codec_ADCCONTROL5, 0x02);

  // set ADC digital volume
  res &= write_reg(Codec_ADCCONTROL8, 0x00);
  res &= write_reg(Codec_ADCCONTROL9, 0x00);

  // recommended ALC setting for VOICE refer to Codec MANUAL
  res &= write_reg(Codec_ADCCONTROL10, 0xEA);
  res &= write_reg(Codec_ADCCONTROL11, 0xC0);
  res &= write_reg(Codec_ADCCONTROL12, 0x12);
  res &= write_reg(Codec_ADCCONTROL13, 0x06);
  res &= write_reg(Codec_ADCCONTROL14, 0xC3);

  /* DAC setting */

  //Power up DAC and Enable LOUT/ROUT
	//res &= writeReg(Codec_DACPOWER, 0x3C,true); //LR 1,2 out enable
	//res &= writeReg(Codec_DACPOWER, 0x0C,true);  //LR 2 out enable
  res &= write_reg(Codec_DACPOWER, 0x3C);

  // SFI setting (i2s mode/16 bit)
  //res &= writeReg(Codec_DACCONTROL1,0x00);// (I2S – 24Bit)
  res &= write_reg(Codec_DACCONTROL1, 0x18);

  //Select MCLK / LRCK ratio for	DAC
	//res &= writeReg(Codec_DACCONTROL2,0x02);// (256 single speed)
	//res &= writeReg(Codec_DACCONTROL2,0x03);// (384, single speed)
  res &= write_reg(Codec_DACCONTROL2, 0x02);

  // unmute codec
  res &= write_reg(Codec_DACCONTROL3, 0x00);

  // set DAC digital volume
  res &= write_reg(Codec_DACCONTROL4, 0x00);
  res &= write_reg(Codec_DACCONTROL5, 0x00);

  //Enable 44.1kHz Deemphasis on single speed, enable click free on power up n down
	//res &= writeReg(Codec_DACCONTROL6,0x88);// (0dB)	

  res &= write_reg(Codec_DACCONTROL7, 0x00);

  // Setup Mixer
  // (reg[16] 1B mic Amp, 0x09 direct;[reg 17-20] 0x90 DAC, 0x50 Mic Amp)
  
  res &= write_reg(Codec_DACCONTROL16, 0x09); //left in select for out mix: L-ADC-IN, Right in select for outmix: R-ADC-IN
  res &= write_reg(Codec_DACCONTROL17, 0x50); //left mixer input from left dac only, gain = 0dB
  res &= write_reg(Codec_DACCONTROL18, 0x38);  //??
  res &= write_reg(Codec_DACCONTROL19, 0x38);  //??
  res &= write_reg(Codec_DACCONTROL20, 0x50); //right mixer input from right dac only, gain = 0dB

  // set Lout/Rout Volume -45db
  res &= write_reg(Codec_DACCONTROL24, 0x00); // L1 (0dB)
  res &= write_reg(Codec_DACCONTROL25, 0x00); // R1 (0dB)
  res &= write_reg(Codec_DACCONTROL26, 0x00); // L2 (0dB)
  res &= write_reg(Codec_DACCONTROL27, 0x00); // R2 (0dB)

  //optimize A/D conversion for 1/4 Vrms range
	//optimizeConversion(2);

  /* Power up DEM and STM */
  res &= write_reg(Codec_CHIPPOWER, 0x00);

  return res;
}

// Select output sink
// OUT1 -> Select Line OUTL/R1
// OUT2 -> Select Line OUTL/R2
// OUTALL -> Enable ALL
bool Codec::outputSelect(outsel_t _sel) {
  bool res = true;
  if (_sel == OUTALL)
    res &= write_reg(Codec_DACPOWER, 0x3C);
  else if (_sel == OUT1)
    res &= write_reg(Codec_DACPOWER, 0x30);
  else if (_sel == OUT2)
    res &= write_reg(Codec_DACPOWER, 0x0C);
  _outSel = _sel;
  return res;
}

// Select input source
// IN1     -> Select Line IN L/R 1
// IN2     -> Select Line IN L/R 2
// IN1DIFF -> differential IN L/R 1
// IN2DIFF -> differential IN L/R 2
bool Codec::inputSelect(insel_t sel) {
  bool res = true;
  if (sel == IN1)
    res &= write_reg(Codec_ADCCONTROL2, 0x00);
  else if (sel == IN2)
    res &= write_reg(Codec_ADCCONTROL2, 0x50);
  else if (sel == IN1DIFF) {
    res &= write_reg(Codec_ADCCONTROL2, 0xF0);
    res &= write_reg(Codec_ADCCONTROL3, 0x00);
  } else if (sel == IN2DIFF) {
    res &= write_reg(Codec_ADCCONTROL2, 0xF0);
    res &= write_reg(Codec_ADCCONTROL3, 0x80);
  }
  _inSel = sel;
  return res;
}

// mute Output
bool Codec::DACmute(bool mute) {
  uint8_t _reg;
  _reg = read_reg(Codec_ADCCONTROL1);
  bool res = true;
  if (mute)
    res &= write_reg(Codec_DACCONTROL3, _reg | 0x04);
  else
    res &= write_reg(Codec_DACCONTROL3, _reg & ~(0x04));
  return res;
}

// set bits per sample as BITS_PER_SAMPLE_16, or BITS_PER_SAMPLE_24
bool Codec::setBitsPerSample() {
  bool res = true;
  if (BITS_PER_SAMPLE == 16)
    { res &= write_reg(Codec_ADCCONTROL4, 0X0C); }
  else if (BITS_PER_SAMPLE == 24)
    { res &= write_reg(Codec_ADCCONTROL4, 0X20); }
  return res;
}

// set output volume max is 33
bool Codec::setOutputVolume(uint8_t vol) {
  if (vol > 33) vol = 33;
  bool res = true;
  if (_outSel == OUTALL || _outSel == OUT1) {
    res &= write_reg(Codec_DACCONTROL24, vol);  // LOUT1VOL
    res &= write_reg(Codec_DACCONTROL25, vol);  // ROUT1VOL
  } else if (_outSel == OUTALL || _outSel == OUT2) {
    res &= write_reg(Codec_DACCONTROL26, vol);  // LOUT2VOL
    res &= write_reg(Codec_DACCONTROL27, vol);  // ROUT2VOL
  }
  return res;
}

uint8_t Codec::getOutputVolume() {
  static uint8_t _reg;
  _reg = read_reg(Codec_DACCONTROL24);
  return _reg;
}

// set input gain max is 8 +24db
bool Codec::setInputGain(uint8_t gain) {
  if (gain > 8) gain = 8;
  bool res = true;
  gain = (gain << 4) | gain;
  res &= write_reg(Codec_ADCCONTROL1, gain);
  return res;
}

uint8_t Codec::getInputGain() {
  static uint8_t _reg;
  _reg = read_reg(Codec_ADCCONTROL1);
  _reg = _reg & 0x0F;
  return _reg;
}

// Recommended ALC setting from User Guide
// DISABLE -> Disable ALC
// GENERIC -> Generic Mode
// VOICE   -> Voice Mode
// MUSIC   -> Music Mode
bool Codec::setALCmode(alcmodesel_t alc) {
  bool res = true;

  // generic ALC setting
  uint8_t ALCSEL = 0b11;       // stereo
  uint8_t ALCLVL = 0b0011;     //-12db
  uint8_t MAXGAIN = 0b111;     //+35.5db
  uint8_t MINGAIN = 0b000;     //-12db
  uint8_t ALCHLD = 0b0000;     // 0ms
  uint8_t ALCDCY = 0b0101;     // 13.1ms/step
  uint8_t ALCATK = 0b0111;     // 13.3ms/step
  uint8_t ALCMODE = 0b0;       // ALC
  uint8_t ALCZC = 0b0;         // ZC off
  uint8_t TIME_OUT = 0b0;      // disable
  uint8_t NGAT = 0b1;          // enable
  uint8_t NGTH = 0b10001;      //-51db
  uint8_t NGG = 0b00;          // hold gain
  uint8_t WIN_SIZE = 0b00110;  // default

  if (alc == DISABLE)
    ALCSEL = 0b00;
  else if (alc == MUSIC) {
    ALCDCY = 0b1010;  // 420ms/step
    ALCATK = 0b0110;  // 6.66ms/step
    NGTH = 0b01011;   // -60db
  } else if (alc == VOICE) {
    ALCLVL = 0b1100;  // -4.5db
    MAXGAIN = 0b101;  // +23.5db
    MINGAIN = 0b010;  // 0db
    ALCDCY = 0b0001;  // 820us/step
    ALCATK = 0b0010;  // 416us/step
    NGTH = 0b11000;   // -40.5db
    NGG = 0b01;       // mute ADC
    res &= write_reg(Codec_ADCCONTROL1, 0x77);
  }
  res &= write_reg(Codec_ADCCONTROL10, ALCSEL << 6 | MAXGAIN << 3 | MINGAIN);
  res &= write_reg(Codec_ADCCONTROL11, ALCLVL << 4 | ALCHLD);
  res &= write_reg(Codec_ADCCONTROL12, ALCDCY << 4 | ALCATK);
  res &= write_reg(Codec_ADCCONTROL13,
                   ALCMODE << 7 | ALCZC << 6 | TIME_OUT << 5 | WIN_SIZE);
  res &= write_reg(Codec_ADCCONTROL14, NGTH << 3 | NGG << 2 | NGAT);

  return res;
}

// MIXIN1 – direct IN1 (default)
// MIXIN2 – direct IN2
// MIXRES – reserved codec
// MIXADC – ADC/ALC input (after mic amplifier)
bool Codec::mixerSourceSelect(mixsel_t LMIXSEL, mixsel_t RMIXSEL) {
  bool res = true;
  uint8_t _reg;
  _reg = (LMIXSEL << 3) | RMIXSEL;
  res &= write_reg(Codec_DACCONTROL16, _reg);
  return res;
}

// LD/RD = DAC(i2s), false disable, true enable
// LI2LO/RI2RO from mixerSourceSelect(), false disable, true enable
// LOVOL = gain, 0 -> 6db, 1 -> 3db, 2 -> 0db, higher will attenuate
bool Codec::mixerSourceControl(bool LD2LO, bool LI2LO, uint8_t LI2LOVOL,
                                bool RD2RO, bool RI2RO, uint8_t RI2LOVOL) {
  bool res = true;
  uint8_t _regL, _regR;
  if (LI2LOVOL > 7) LI2LOVOL = 7;
  if (RI2LOVOL > 7) RI2LOVOL = 7;
  _regL = (LD2LO << 7) | (LI2LO << 6) | (LI2LOVOL << 3);
  _regR = (RD2RO << 7) | (RI2RO << 6) | (RI2LOVOL << 3);
  res &= write_reg(Codec_DACCONTROL17, _regL);
  res &= write_reg(Codec_DACCONTROL20, _regR);
  return res;
}

// Mixer source control
// DACOUT -> Select Sink From DAC
// SRCSEL -> Select Sink From SourceSelect()
// MIXALL -> Sink DACOUT + SRCSEL
bool Codec::mixerSourceControl(mixercontrol_t mix) {
  bool res = true;
  if (mix == DACOUT)
    mixerSourceControl(true, false, 2, true, false, 2);
  else if (mix == SRCSELOUT)
    mixerSourceControl(false, true, 2, false, true, 2);
  else if (mix == MIXALL)
    mixerSourceControl(true, true, 2, true, true, 2);
  return res;
}

// true -> analog out = analog in
// false -> analog out = DAC(i2s)
bool Codec::analogBypass(bool bypass) {
  bool res = true;
  if (bypass) {
    if (_inSel == IN1)
      mixerSourceSelect(MIXIN1, MIXIN1);
    else if (_inSel == IN2)
      mixerSourceSelect(MIXIN2, MIXIN2);
    mixerSourceControl(false, true, 2, false, true, 2);
  } else {
    mixerSourceControl(true, false, 2, true, false, 2);
  }
  return res;
}

void Codec::optimizeConversion(int range)
{
	int ingain[]={0, 2, 4, 6, 8}; //0db, 6dB, 12dB, 18dB, 24dB
	int outvol[]= {30, 26, 22, 18, 14}; //0db, -6dB, -12dB, -18dB, -24dB
	if(range<0) range = 0;
	if(range>4) range = 4;
	setOutputVolume(outvol[range]);
	setInputGain(ingain[range]);
}
	
//get and set microphone gain (0:0dB,1-7:30dB-48dB)
uint8_t Codec::getMicGain()
{
	if(getInputMode()==1) //input mode = LMIC
		return (0x0F & read_reg(Codec_ADCCONTROL1));
	else return 0;
}

bool Codec::setMicGain(uint8_t gain)
{	
	if(getInputMode()==1) //input mode = LMIC
	{
		if(gain > 8) gain = 8;
    static uint8_t temp;
		temp = read_reg(Codec_ADCCONTROL1);
		temp = temp & 0xF0;
		temp = temp | gain;
		return write_reg(Codec_ADCCONTROL1, temp); 
	}
	else return false;
}

int Codec::getMicNoiseGate()
{
	if(getInputMode()==1)
	{
    static uint8_t temp;
		temp = read_reg(Codec_ADCCONTROL14);
		temp = temp >> 3;
		return (int) temp;
	}
	else return 0;
}

bool Codec::setMicNoiseGate(int gate)
{
	if(getInputMode()==1) //input mode = IM_LMIC
	{
		if(gate > 32) gate = 32;
		if(gate>0)
		{
			uint8_t temp = ((gate-1) << 3) | 1;
			return write_reg(Codec_ADCCONTROL14,temp); 
		}
		else //turn off the noise gate at gate = 0
		{
			return write_reg(Codec_ADCCONTROL14,0); 
		}
	}
	else return false;
}

/*
//bypassed the analog input to the output, disconnect the digital i/o 
bool Codec::analogBypass(bool bypass, bypass_mode_t bm)
{
	bool res = true;
	if(bypass)
	{
		if((bm==XBM_LR)||(bm==XBM_L))
			*muteLeftAdcIn = true;
		if((bm==XBM_LR)||(bm==XBM_R))
			*muteRightAdcIn= true;
		
		if((bm==XBM_LR)||(bm==XBM_L))
			res &= write_reg(Codec_DACCONTROL17,0x50); //disable ldac, enable lin, gain = 0dB
		if((bm==XBM_LR)||(bm==XBM_R))
			res &= write_reg(Codec_DACCONTROL20,0x50); //disable rdac, enable rin, gain = 0dB
	}
	else
	{
		if((bm==XBM_LR)||(bm==XBM_L))
			*muteLeftAdcIn = false;
		if((bm==XBM_LR)||(bm==XBM_R))
			*muteRightAdcIn= false;
		
		if((bm==XBM_LR)||(bm==XBM_L))
			res &= write_reg(Codec_DACCONTROL17,0x90); //enable ldac,disable lin, gain = 0dB
		if((bm==XBM_LR)||(bm==XBM_R))
			res &= write_reg(Codec_DACCONTROL20,0x90); //enable rdac,disable rin, gain = 0dB
	}	
	return res;
}
*/

//bypassed the analog input to the output, disconnect the digital input, preserve the digital output connection
bool Codec::analogSoftBypass(bool bypass, bypass_mode_t bm)
{
	bool res = true;
	if(bypass)
	{
		if((bm==XBM_LR)||(bm==XBM_L))
			*muteLeftAdcIn = true;
		if((bm==XBM_LR)||(bm==XBM_R))
			*muteRightAdcIn= true;
		
		if((bm==XBM_LR)||(bm==XBM_L))
			res &= write_reg(Codec_DACCONTROL17,0xD0); //enable ldac, enable lin, gain = 0dB
		if((bm==XBM_LR)||(bm==XBM_R))
			res &= write_reg(Codec_DACCONTROL20,0xD0); //enable rdac, enable rin, gain = 0dB
	}
	else
	{
		if((bm==XBM_LR)||(bm==XBM_L))
			*muteLeftAdcIn = false;
		if((bm==XBM_LR)||(bm==XBM_R))
			*muteRightAdcIn= false;
		
		if((bm==XBM_LR)||(bm==XBM_L))
			res &= write_reg(Codec_DACCONTROL17,0x90); //enable ldac,disable lin, gain = 0dB
		if((bm==XBM_LR)||(bm==XBM_R))
			res &= write_reg(Codec_DACCONTROL20,0x90); //enable rdac,disable rin, gain = 0dB
	}
	return res;
}

